<?php

namespace App\modules\Mis\Services;

use App\modules\Mis\Models\BikrMaschinendaten;
use App\modules\Mis\Models\BikrMaschinenstundensaetze;
use App\modules\Mis\Models\BikrParameter;

class KlrCalcService {

    /**
     * Berechnet die Maschinenstundensaetze neu. Jede Berechnung bekommt ein neues Gueltigkeitsdatum.
     * Berechnungen jeder MaschinenID davor werden auf aktiv=0 gesetzt:
     *
     *      1. alle Datensaetze (wenn existent) der aktuellen MaschinenID weden auf aktiv=0 gesetzt
     *      2. dann wird neue Berechnung gespeichert mit aktiv=1
     *
     * @param $maschine_id
     * @return true
     * @throws \Throwable
     */
    public function clacStundensatz($maschine_id)
    {
        // read maschine
        $maschine = BikrMaschinendaten::find($maschine_id);
        // read parameter
        $p = BikrParameter::all();
        $parameter = $this->prepareParameter($p);

        $calcMatrix = [
            'maschine_id' => $maschine->id,
            'kalk_abschreibung' => 0,
            'abschreibung_h' => 0,
            'raumkosten_h' => 0,
            'energiekosten_h' => 0,
            'betriebskosten_h' => 0,
            'summe_a' => 0,
            'stueck_a' => 0,
            'summe_b' => 0,
            'stueck_b' => 0,
            'summe_c' => 0,
            'stueck_c' => 0,
            'gueltig_ab' => date('Ymd'),
            'aktiv' => 1,
        ];

        // calc Schema
        // kalk.Abschreibung pa = (Wiederbeschaffungswert - Restbuchwert) / reale Nutzungsdauer
        $calcMatrix['abschreibung_k'] = round(
            ($maschine->wiederbeschaffungswert - $maschine->restbuchwert) / $maschine->nutzungsdauer,
        4
        );

        // Abschreibung h = (Wiederbeschaffungswert - Restbuchwert) / (reale Nutzungsdauer * Laufstunden)
        $calcMatrix['abschreibung_h'] = round(
            ($maschine->wiederbeschaffungswert - $maschine->restbuchwert) / ($maschine->nutzungsdauer * $parameter['Laufstunden']),
        4
        );

        // Raumkosten h = (Raumbedarf * Raumkostensatz) / Laufstunden
        $calcMatrix['raumkosten_h'] = round(
            ($maschine->raumbedarf * $parameter['Raumkostensatz']) / $parameter['Laufstunden'],
        4
        );

        // Energiekosten h = Anschlusswert * Energiekostensatz
        $calcMatrix['energie_h'] = round(
            $maschine->anschlussleistung * $parameter['Energiekostensatz'],
        4
        );

        // Betriebskosten h = (Instandhaltung + Betriebskosten) / Laufstunden
        $calcMatrix['betrieb_h'] = round(
            ($maschine->instandhaltungskosten + $maschine->betriebskosten) / $parameter['Laufstunden'],
        4
        );

        // Summe_a_100% = Abschreibung h + Raumkosten h + Energiekosten h + Betriebskosten h
        $calcMatrix['auslastung_a'] = $parameter['Auslastung_a'];
        $calcMatrix['summe_a'] = $calcMatrix['abschreibung_h'] + $calcMatrix['raumkosten_h'] + $calcMatrix['energie_h'] + $calcMatrix['betrieb_h'];

        // Stückkosten_a = Summe_a_100% / (Kapazität_minute * 60)
        $calcMatrix['stueck_a'] = round(
            $calcMatrix['summe_a'] / ($maschine->kapazitaet_minute * 60),
        5
        );

        // Summe_b_75% = Summe_a_100% * 75%
        $calcMatrix['auslastung_b'] = $parameter['Auslastung_b'];
        $calcMatrix['summe_b'] = round(
            $calcMatrix['summe_a'] / ($parameter['Auslastung_b'] / 100),
        4
        );

        // Stückkosten_b = Summe_b_100% / (Kapazität_minute * 60)
        $calcMatrix['stueck_b'] = round(
            $calcMatrix['summe_b'] / ($maschine->kapazitaet_minute * 60),
            5
        );

        // Summe_c_50% = Summe_a_100% * 50%
        $calcMatrix['auslastung_c'] = $parameter['Auslastung_c'];
        $calcMatrix['summe_c'] = round(
            $calcMatrix['summe_a'] / ($parameter['Auslastung_c'] / 100),
            4
        );

        // Stückkosten_c = Summe_c_100% / (Kapazität_minute * 60)
        $calcMatrix['stueck_c'] = round(
            $calcMatrix['summe_c'] / ($maschine->kapazitaet_minute * 60),
            5
        );

        \DB::beginTransaction();
        try {
            if (BikrMaschinenstundensaetze::where('maschine_id', '=', $maschine->id)->count() > 0) {
                BikrMaschinenstundensaetze::where('maschine_id', '=', $maschine->id)->update(['aktiv' => 0]);
            }
            BikrMaschinenstundensaetze::create($calcMatrix);

            \DB::commit();
        } catch (\Exception $e) {
            \DB::rollBack();
        }

        return true;
    }

    /**
     * Format the Parameter from Database into an associative Array.
     * ```
     * $preparedParameter[$parameter->name] = $parameter->value_numeric;
     * ```
     *
     * @param $parameter
     * @return array
     */
    protected function prepareParameter($parameter)
    {
        $preparedParameter = [];

        foreach ($parameter as $parameter) {
            $preparedParameter[$parameter->name] = $parameter->value_numeric;
        }
        return $preparedParameter;
    }
}
