<?php

namespace App\modules\Mis\Controllers;

use App\Http\Controllers\Controller;
use App\modules\Core\Traits\ModuleHelperTrait;
use App\modules\Mis\Models\BikrBilanz;
use App\modules\Mis\Models\BikrMaschinendaten;
use App\modules\Mis\Models\BikrMaschinenstundensaetze;
use App\modules\Mis\Models\BikrParameter;
use App\modules\Mis\Services\KlrCalcService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Gate;

class MisController extends Controller
{
    use ModuleHelperTrait;

    /**
     * Navigation
     */

    public function index() { return $this->showModuleList(['mis']); }

    public function stat() { return $this->showModuleList(['mis','stat']); }
    public function statEinkauf() { return $this->showFnList(['mis','stat', 'ek']); }
    public function statVerkauf() { return $this->showFnList(['mis','stat', 'vk']); }
    public function statBestand() { return $this->showFnList(['mis','stat', 'bestand']); }

    // Kosten- und Leistungsrechnung Navigationsmodul
    public function klr() { return $this->showModuleList(['mis','klr']); }
    public function klrStammdaten() { return $this->showFnList(['mis','klr','stammdaten']); }
    public function klrMaschinenkalkulation() { return $this->showFnList(['mis','klr','maschinenkalkulation']); }
    public function klrProduktlinienkalkulation() { return $this->showFnList(['mis','klr','produktlinienkalkulation']); }
    public function klrAufschlagskalkulation() { return $this->showFnList(['mis','klr','aufschlagskalkulation']); }
    public function klrRentabilitaet() { return $this->showFnList(['mis','klr','rentabilitaet']); }


    /**
     * KLR.Stammdaten.Bilanz
     */
    public function klrStammBilanzIndex()
    {
        $tableDef = BikrBilanz::indexColumnSettings();
        $data = BikrBilanz::all(array_keys($tableDef['columns']));

        return view('mis::klr.stammdaten.bilanz_index', compact('data', 'tableDef'));
    }

    public function klrStammBilanzCreate()
    {
        $colDef = BikrBilanz::createColumnSettings();
        return view('mis::klr.stammdaten.bilanz_create', compact('colDef'));
    }

    public function klrStammBilanzShow(BikrBilanz $bilanz)
    {
        return view('mis::klr.stammdaten.bilanz_show', compact('bilanz'));
    }

    public function klrStammBilanzEdit(BikrBilanz $bilanz)
    {
        return view('mis::klr.stammdaten.bilanz_edit', compact('bilanz'));
    }

    public function klrStammBilanzDelete(BikrBilanz $bilanz)
    {
        $bilanz->delete();
        return redirect('/mis/klr/stammdaten/bilanz');
    }

    public function klrStammBilanzStore(\Request $request)
    {
        // validation
        $attributes = request()->validate([
            'jahr' => ['required', 'numeric', 'max:'.date('Y'), 'unique:bikr_bilanzen,jahr'],
            'lohn' => ['required', 'numeric'],
            'sozialabgaben' => ['required', 'numeric', 'gt:0'],
            'abschreibungen' => ['required', 'numeric', 'gt:0'],
            'aufwaende' => ['required', 'numeric', 'gt:0'],
            'zinsen' => ['required', 'numeric', 'gt:0'],
            'vorraete' => ['required', 'numeric', 'gt:0'],
            'personalzahl' => ['required', 'numeric'],
        ]);

        BikrBilanz::create($attributes);

        return redirect('/mis/klr/stammdaten/bilanz');
    }

    public function klrStammBilanzUpdate(\Request $request, BikrBilanz $bilanz)
    {
        Gate::authorize('update', $bilanz);
        // validation
        request()->validate([
            'jahr' => ['required', 'numeric', 'max:'.date('Y'), 'unique:bikr_bilanzen,jahr,'.$bilanz->id],
            'lohn' => ['required', 'numeric'],
            'sozialabgaben' => ['required', 'numeric', 'gt:0'],
            'abschreibungen' => ['required', 'numeric', 'gt:0'],
            'aufwaende' => ['required', 'numeric', 'gt:0'],
            'zinsen' => ['required', 'numeric', 'gt:0'],
            'vorraete' => ['required', 'numeric', 'gt:0'],
            'personalzahl' => ['required', 'numeric'],
        ]);

        $bilanz->update([
            'jahr' => request('jahr'),
            'lohn' => request('lohn'),
            'sozialabgaben' => request('sozialabgaben'),
            'abschreibungen' => request('abschreibungen'),
            'aufwaende' => request('aufwaende'),
            'zinsen' => request('zinsen'),
            'vorraete' => request('vorraete'),
            'personalzahl' => request('personalzahl'),
        ]);

        return redirect()->route('mis.klr.stammdaten.bilanz', ['lastChanged' => $bilanz->id]);
    }



    /**
     * KLR.Stammdaten.Maschine
     */
    public function klrStammMaschineIndex()
    {
        $tableDef = BikrMaschinendaten::indexColumnSettings();
        $data = BikrMaschinendaten::all(array_keys($tableDef['columns']));

        return view('mis::klr.stammdaten.maschine_index', compact('data', 'tableDef'));
    }

    public function klrStammMaschineCreate()
    {
        $colDef = BikrMaschinendaten::createColumnSettings();
        return view('mis::klr.stammdaten.maschine_create', compact('colDef'));
    }

    public function klrStammMaschineShow(BikrMaschinendaten $maschine)
    {
        return view('mis::klr.stammdaten.maschine_show', compact('maschine'));
    }

    public function klrStammMaschineEdit(BikrMaschinendaten $maschine)
    {
        return view('mis::klr.stammdaten.maschine_edit', compact('maschine'));
    }

    public function klrStammMaschineStore(Request $request)
    {
        // validation
        $requestData = $request->all();
        // modify fields
        $requestData['ms_faktor'] = trim(str_replace(',', '.', $requestData['ms_faktor']));
        $request->replace($requestData);

        $request->validate([
            'name' => ['required', 'string', 'min:3'],
            'bezeichnung' => ['required', 'string', 'min:3'],
            'anschaffungswert' => ['required', 'numeric'],
            'restbuchwert' => ['required', 'numeric'],
            'wiederbeschaffungswert' => ['required', 'numeric'],
            'beschaffungsjahr' => ['required', 'numeric', 'max:'.date('Y')],
            'nutzungsdauer' => ['required', 'numeric', 'gt:0'],
            'raumbedarf' => ['required', 'numeric', 'gt:0'],
            'anschlussleistung' => ['required', 'numeric', 'gt:0'],
            'kapazitaet' => ['required', 'numeric', 'gt:0'],
            'betriebskosten' => ['required', 'numeric', 'gt:0'],
            'instandhaltungskosten' => ['required', 'numeric', 'gt:0'],
            'ms_faktor' => ['nullable', 'numeric'],
            'faktor_beschreibung' => ['nullable', 'string'],
        ]);

        BikrMaschinendaten::create([
            'name' => request('name'),
            'bezeichnung' => request('bezeichnung'),
            'beschreibung' => request('beschreibung'),
            'anschaffungswert' => request('anschaffungswert'),
            'restbuchwert' => request('restbuchwert'),
            'wiederbeschaffungswert' => request('wiederbeschaffungswert'),
            'beschaffungsjahr' => request('beschaffungsjahr'),
            'nutzungsdauer' => request('nutzungsdauer'),
            'raumbedarf' => request('raumbedarf'),
            'anschlussleistung' => request('anschlussleistung'),
            'kapazitaet' => request('kapazitaet'),
            'betriebskosten' => request('betriebskosten'),
            'instandhaltungskosten' => request('instandhaltungskosten'),
            'ms_faktor' => request('ms_faktor'),
            'faktor_beschreibung' => request('faktor_beschreibung'),
        ]);

        return redirect('/mis/klr/stammdaten/maschine');
    }

    public function klrStammMaschineUpdate(Request $request, BikrMaschinendaten $maschine)
    {
        // validation
        $requestData = $request->all();
        // modify fields
        $requestData['ms_faktor'] = trim(str_replace(',', '.', $requestData['ms_faktor']));
        $request->replace($requestData);

        $request->validate([
            'name' => ['required', 'string', 'min:3'],
            'bezeichnung' => ['required', 'string', 'min:3'],
            'anschaffungswert' => ['required', 'numeric'],
            'restbuchwert' => ['required', 'numeric'],
            'wiederbeschaffungswert' => ['required', 'numeric'],
            'beschaffungsjahr' => ['required', 'numeric', 'max:'.date('Y')],
            'nutzungsdauer' => ['required', 'numeric', 'gt:0'],
            'raumbedarf' => ['required', 'numeric', 'gt:0'],
            'anschlussleistung' => ['required', 'numeric', 'gt:0'],
            'kapazitaet_jahr' => ['required', 'numeric', 'gt:0'],
            'kapazitaet_minute' => ['required', 'numeric', 'gt:0'],
            'betriebskosten' => ['required', 'numeric', 'gt:0'],
            'instandhaltungskosten' => ['required', 'numeric', 'gt:0'],
            'ms_faktor' => ['numeric', 'nullable'],
            'faktor_beschreibung' => ['string', 'nullable'],
        ]);

        $maschine->update([
            'name' => request('name'),
            'bezeichnung' => request('bezeichnung'),
            'beschreibung' => request('beschreibung'),
            'anschaffungswert' => request('anschaffungswert'),
            'restbuchwert' => request('restbuchwert'),
            'wiederbeschaffungswert' => request('wiederbeschaffungswert'),
            'beschaffungsjahr' => request('beschaffungsjahr'),
            'nutzungsdauer' => request('nutzungsdauer'),
            'raumbedarf' => request('raumbedarf'),
            'anschlussleistung' => request('anschlussleistung'),
            'kapazitaet' => request('kapazitaet'),
            'betriebskosten' => request('betriebskosten'),
            'instandhaltungskosten' => request('instandhaltungskosten'),
            'ms_faktor' => request('ms_faktor'),
            'faktor_beschreibung' => request('faktor_beschreibung'),
        ]);

        return redirect()->route('mis.klr.stammdaten.maschine', ['lastChanged' => $maschine->id]);
    }



    /**
     * KLR.Stammdaten.Parameter
     */
    public function klrStammParamIndex()
    {
        $tableDef = BikrParameter::indexColumnSettings();
        $data = BikrParameter::all(array_keys($tableDef['columns']));
        return view('mis::klr.stammdaten.parameter_index', compact('data', 'tableDef'));
    }

    public function klrStammParamCreate()
    {
        $colDef = BikrParameter::createColumnSettings();
        return view('mis::klr.stammdaten.parameter_create', compact('colDef'));
    }

    public function klrStammParamShow(BikrParameter $parameter)
    {
        return view('mis::klr.stammdaten.parameter_show', compact('parameter'));
    }

    public function klrStammParamEdit(BikrParameter $parameter)
    {
        return view('mis::klr.stammdaten.parameter_edit', compact('parameter'));
    }

    public function klrStammParamStore(\Request $request)
    {
        // validation
        request()->validate([
            'name' => ['required', 'string', 'min:3'],
            'bezeichnung' => ['required', 'string', 'min:3'],
            #'beschreibung' => ['string'],
            'value_numeric' => ['sometimes', 'numeric'],
            #'value_string' => ['string'],
        ]);

        BikrParameter::create([
            'name' => request('name'),
            'bezeichnung' => request('bezeichnung'),
            'beschreibung' => request('beschreibung'),
            'value_numeric' => request('value_numeric'),
            'value_string' => request('value_string') ?? '',
        ]);

        return redirect('/mis/klr/stammdaten/parameter');
    }

    public function klrStammParamUpdate(\Request $request, BikrParameter $parameter)
    {
        // validation
        request()->validate([
            'name' => ['required', 'string', 'min:3'],
            'bezeichnung' => ['required', 'string', 'min:3'],
            #'beschreibung' => ['sometimes', 'required', 'string'],
            'value_numeric' => ['sometimes', 'required', 'numeric'],
            #'value_string' => ['sometimes', 'string'],
        ]);

        $parameter->update([
            'name' => request('name'),
            'bezeichnung' => request('bezeichnung'),
            'beschreibung' => request('beschreibung'),
            'value_numeric' => request('value_numeric') ?? 0.00,
            'value_string' => request('value_string') ?? '',
        ]);

        return redirect()->route('mis.klr.stammdaten.parameter', ['lastChanged' => $parameter->id]);
    }


    /**
     * KLR.Maschinenkalkulation
     */
    public function klrMaschinenkalkulationDatenblattIndex()
    {
        $tableDef = BikrMaschinendaten::selectColumnSettings();
        #$data = BikrMaschinendaten::all(array_keys($tableDef['columns']));
        $data = BikrMaschinendaten::with('stundensatz')->get();
        return view('mis::klr.maschinenkalkulation.datenblatt_index', compact('data', 'tableDef'));
    }

    public function klrMaschinenkalkulationDatenblaetterIndex()
    {
        $tableDef = BikrMaschinenstundensaetze::allHourlyRatesColumns();
        $data = BikrMaschinenstundensaetze::where('aktiv', '=', 1)
            ->with('maschine')
            ->orderBy('maschine_id')->get();

        return view(
            'mis::klr.maschinenkalkulation.datenblaetter_index',
            compact('data', 'tableDef')
        );
    }

    public function klrMaschinenkalkulationDatenblattShow(BikrMaschinendaten $maschine)
    {
        $stundensatz = BikrMaschinenstundensaetze::where('maschine_id', '=', $maschine->id)->where('aktiv', '=', 1)->first();
        return view('mis::klr.maschinenkalkulation.datenblatt_show', compact('maschine', 'stundensatz'));
    }

    /**
     * Brechnet eine Maschine neu
     * @param  BikrMaschinendaten  $maschine
     * @return string
     */
    public function klrMaschinenkalkulationBerechnen(BikrMaschinendaten $maschine)
    {
        $klrCalcService = new KlrCalcService();
        $result = $klrCalcService->clacStundensatz($maschine->id);
        $stundensatz = BikrMaschinenstundensaetze::where('maschine_id', '=', $maschine->id)->where('aktiv', '=', 1)->first();
        return view('mis::klr.maschinenkalkulation.datenblatt_show', compact('maschine', 'stundensatz'));
    }

    /**
     * Berechnet alle Maschinen neu
     * @return string
     */
    public function klrMaschinenkalkulationBerechnenAll()
    {
        $klrCalcService = new KlrCalcService();
        $maschinen = BikrMaschinendaten::all();
        foreach ($maschinen as $maschine) {
            $result = $klrCalcService->clacStundensatz($maschine->id);
        }

        $data = BikrMaschinenstundensaetze::where('aktiv', '=', 1)
            ->with('maschine')
            ->orderBy('maschine_id')->get();
        $tableDef = BikrMaschinenstundensaetze::allHourlyRatesColumns();
        return view('mis::klr.maschinenkalkulation.datenblaetter_index', compact('data', 'tableDef'));
    }

}
