<?php

namespace App\modules\Core\Traits;

trait ModuleHelperTrait
{
    /**
     * Find all activated Modules and list der lowercase Name into an Array.
     * TODO: create a Checkmethode to find active Modules
     *
     * @return string[]
     */
    public function getActiveModule() : array
    {
        $active_modules = explode(',', config('core.active_modules'));
        return $active_modules;

//        return [
//            'neo',
//            'mis',
//            'admin'
//        ];
    }


    /**
     * Read all Module Navigation Info from their module-config-files and organise all Info
     * in a structured Array.
     * Return-Format:
     * ```
     * [
     *      'meta' => [
     *          'name' => 'Mis',
     *          'desc' => 'Management Information System',
     *          'link' => 'mis',
     *          'icon' => 'fa-chart-line'
     *      ],
     *      'functions' => [
     *          'controlling' => [
     *              'meta' => [
     *                 'name' => 'Controlling',
     *                  'desc' => 'Werkzeuge zum Controlling der Unternehmensfinanzen',
     *                  'link' => 'controlling',
     *                  'icon' => 'fa-percent'
     *              ],
     *              'functions' => [],
     *          ],
     *      ],
     *      ...
     * ]
     * ```
     *
     * @param  array  $activeModules
     *                ['modul1', 'modul2', ...] as lowercase
     * @return array
     */
    public function getModuleNavigationInfo(Array $activeModules): array
    {
        $modules = [];
        foreach ($activeModules as $module) {
            $_m = strtolower($module);
            if (isset(config($_m)['navigation'])) {
                $modules[$_m] = config($_m)['navigation'];
            }
        }

        return [
            'name' => config('core.subtitle'),
            'title' => config('core.title'),
            'version' => config('core.version'),
            'state' => config('core.state'),
            'activeModules' => $modules
        ];
    }

    public function getPermissions($modulName = ''): array
    {
        $activeModules = $this->getActiveModule();
        $navigationInfos = $this->getModuleNavigationInfo($activeModules);
        $permissions = [];

        if($modulName !== '') {
            $this->scanPermission($navigationInfos['activeModules'][$modulName], $permissions);
        } else {
            foreach ($activeModules as $modulName) {
                $this->scanPermission($navigationInfos['activeModules'][$modulName], $permissions);
            }
        }

        return $permissions;
    }

    public function getAllModels()
    {
        $activeModules = $this->getActiveModule();
        $models = [];
        foreach ($activeModules as $module) {
            $_m = strtolower($module);
            if (isset(config($_m)['models'])) {
                $models = array_merge($models, config($_m)['models']);
            }
        }
        return $models;
    }


    /**
     * Internal Methods
     */

    protected function getModulePath(array $modulName)
    {
        $moduleInfo = $this->getModuleNavigationInfo($modulName);
        $modulePath = $moduleInfo['activeModules'];
        $cnt = 0;

        foreach ($modulName as $modul) {
            if ($cnt > 0) {
                $modulePath = $modulePath['functions'];
            }
            $modulePath = $modulePath[$modul];
            $cnt++;
        }
        return $modulePath;
    }

    protected function showModuleList(array $modulName)
    {
        $modulePath = $this->getModulePath($modulName);
        return view('core::modulelist', ['moduleInfo' => $modulePath]);
    }

    protected function showFnList($fnName)
    {
        $modulePath = $this->getModulePath($fnName);
        return view('core::fnlist', ['moduleInfo' => $modulePath]);
    }

    protected function scanPermission(Array $navigationInfos, Array &$permissions)
    {
        if (isset($navigationInfos['meta'])) {
            $permission = $this->getPermissionFromMeta($navigationInfos['meta']);
            if ($permission) {
                $permissions[] = $permission;
            }
        }
        if (isset($navigationInfos['functions'])) {
            $permission = $this->getPermissionFromFunction($navigationInfos['functions'], $permissions);
            if ($permission) {
                $permissions[] = $permission;
            }
        }
    }

    /**
     * Permissions derived from the link parameter.
     * Link: modul/sub/action  ->  Permission: modul.sub.action
     */
    protected function getPermissionFromMeta(Array $meta): ?string
    {
        $hasPermission = $meta['permission'] ?? false;
        $permission = null;
        if ($hasPermission) {
            $_token = trim($meta['link'], '/');
            $token = str_replace('/', '.', $_token);
            $permission = $token.'|'.$meta['name'].'|'.$meta['desc'];
        }
        return $permission;
    }

    /**
     * Recursion to scan functions within functions ...
     */
    protected function getPermissionFromFunction(Array $functions, &$permissions)
    {
        foreach ($functions as $function) {
            $this->scanPermission($function, $permissions);
        }
    }
}
