<?php

namespace App\modules\Admin\Controllers;

use App\Http\Controllers\Controller;
use App\Models\Role;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Gate;

class UserAdminController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $data = User::all();
        $tableDef = User::indexColumnSettings();
        return view('admin::user.index', compact('data', 'tableDef'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $roles = Role::all();
        return view('admin::user.create', ['roles' => $roles]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $attributes = $request->validate([
            'name' => ['required'],
            'login' => ['required', 'unique:users', 'min:4'],
            'email' => ['required', 'email', 'max:254'],
            'password' => ['required', 'string', 'min:3', 'confirmed'],
        ]);

        $attributes['status'] = isset($request->status) ? 1 : 0;
        $attributes['email_verified_at'] = date('Y-m-d H:i:s');
        $user = User::create($attributes);
        $user->roles()->sync($request->roles);

        return redirect()->route('admin.user');
    }

    /**
     * Display the specified resource.
     */
    public function show(User $user)
    {
        $roles = $user->roles;
        $groups = $user->groups();
        $permissions = $user->permissions();
        $userPermissions = $user->userPermissions;
        return view(
            'admin::user.show',
            ['user' => $user, 'roles' => $roles, 'groups' => $groups, 'permissions' => $permissions, 'userPermissions' => $userPermissions]
        );
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(User $user)
    {
        $roles = Role::all();
        $groups = $user->groups();
        return view('admin::user.edit', ['user' => $user, 'roles' => $roles, 'groups' => $groups]);
    }

    public function changePassword(Request $request, User $user)
    {
        return view('admin::user.cpwd', compact('user'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, User $user)
    {
        Gate::authorize('update', $user);

        $attributes = $request->validate([
            'name' => ['required'],
//            'login' => ['required', 'unique:users', 'min:4'],
            'email' => ['required', 'email', 'max:254'],
        ]);

        if (!isset($request->status)) {
            $attributes['status'] = 0;
        } else {
            $attributes['status'] = 1;
        }
        $user->update($attributes);
        $user->roles()->sync($request->roles);

        return redirect()->route('admin.user.show', ['lastChanged' => $user->id, 'user' => $user]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function updatePassword(Request $request, User $user)
    {
        $attributes = $request->validate([
            'password' => ['required', 'string', 'min:3', 'confirmed'],
        ]);
        $user->update($attributes);
        return redirect()->route('admin.user', ['lastChanged' => $user->id]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(User $user)
    {
        $user->delete();
        return redirect()->route('admin.user');
    }


    /**
     * Useractions
     */

    public function useraction(Request $request)
    {
        $userAction = $request->get('u_action');
        switch ($userAction) {
            case 'lock':
                break;
            case 'logout':
                return redirect()->route('logout');
                break;
            case 'usersettings':
                return redirect()->route('admin.usersettings');
                break;
        }

        return "useraction";
    }

    public function usersettings()
    {
        $currentUser = \Auth::user();
        return view('admin::usersettings.settings', compact('currentUser'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update_usersettings(Request $request, User $user)
    {
        $attributes = $request->validate([
            'name' => ['required'],
            'email' => ['required', 'email', 'max:254'],
            'password' => ['nullable', 'string', 'min:3', 'confirmed'],
        ]);

        if (is_null($request->get('password'))) {
            unset($attributes['password']);
        }

        $user->update($attributes);
        return redirect()->route('home');
    }
}
